/********************************************************************
 FileName:		Keyboard.c
 Dependencies:	See INCLUDES section
 Processor:		PIC18 or PIC24 USB Microcontrollers
 Hardware:		The code is natively intended to be used on the following
 				hardware platforms: PICDEMEFS USB Demo Board, 
 				PIC18F87J50 FS USB Plug-In Module, or
 				Explorer 16 + PIC24 USB PIM.  The firmware may be
 				modified for use on other USB platforms by editing the
 				HardwareProfile.h file.
 Complier:  	Microchip C18 (for PIC18) or C30 (for PIC24)
 Company:		Microchip Technology, Inc.

 Software License Agreement:

 The software supplied herewith by Microchip Technology Incorporated
 (the CompanyE for its PIC Microcontroller is intended and
 supplied to you, the Companys customer, for use solely and
 exclusively on Microchip PIC Microcontroller products. The
 software is owned by the Company and/or its supplier, and is
 protected under applicable copyright laws. All rights are reserved.
 Any use in violation of the foregoing restrictions may subject the
 user to criminal sanctions under applicable laws, as well as to
 civil liability for the breach of the terms and conditions of this
 license.

 THIS SOFTWARE IS PROVIDED IN AN AS ISECONDITION. NO WARRANTIES,
 WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT,
 IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.

********************************************************************
 File Description:

 Change History:
  Rev   Date         Description
  1.0   11/19/2004   Initial release
  2.1   02/26/2007   Updated for simplicity and to use common
                     coding style
  3.0   08/18/2009   Customized for USB keyboard construction set                    
********************************************************************/

#ifndef KEYBOARD_C
#define KEYBOARD_C

/** INCLUDES *******************************************************/
#include "GenericTypeDefs.h"
#include "Compiler.h"
#include "usb_config.h"
#include "./USB/usb_device.h"
#include "usb_device_m.h"
#include "./USB/usb.h"
#include "HardwareProfile.h"
#include "./USB/usb_function_hid.h"
#include <portb.h>
#include <timers.h>

/**       ++++++++++    CHANGE HERE !!  ++++++++++         **/
// EEPROM Map:KEY MATRIX to USB SCANCODE
#pragma romdata eedata1_scn=0xf00000
ROM unsigned char eedata1_values[0x80] =
{ 
0x29,0x26,0x12,0x0f,0x37,0xff,0xff,0xff ,0x60,0x5f,0x63,0x5d,0x5a,0xff,0xff,0xff,
0x61,0x2a,0x53,0x5e,0x5b,0xff,0xff,0xff ,0x55,0x89,0x28,0x56,0x57,0xff,0xff,0xff,
0x2c,0x24,0x18,0x0d,0x10,0x3a,0xff,0xff ,0x52,0x21,0x15,0x09,0x19,0x3b,0xff,0xff,
0x35,0x23,0x1c,0x0b,0x11,0x3c,0xff,0xff ,0x50,0x25,0x0c,0x0e,0x36,0x3d,0xff,0xff,
0x51,0x27,0x13,0x33,0x38,0x3e,0xff,0xff ,0x4f,0x2e,0x30,0x32,0x54,0xff,0xff,0xff,
0xe0,0x2d,0x2f,0x34,0x87,0xff,0xff,0xff ,0xe1,0x20,0x62,0x5c,0x59,0xff,0xff,0xff,
0xff,0xff,0x14,0x04,0x1d,0xff,0xff,0xff ,0x39,0x22,0x17,0x0a,0x05,0xff,0xff,0xff,
0x2b,0x1e,0x1a,0x16,0x1b,0xff,0xff,0xff ,0xe2,0x1f,0x08,0x07,0x06,0xff,0xff,0xff
};
#pragma romdata

// EEPROM Map:Scancode for Option key
#pragma romdata eedata2_scn=0xf00080
ROM unsigned char eedata2_values[0x10] =
{ 
0x35,0xff,0xff,0xff,0xff,0xff,0xff,0xff ,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
};
#pragma romdata

// EEPROM Map:HHK like Fn key
#pragma romdata eedata3_scn=0xf00090
ROM unsigned char eedata3_values[0x40] =
{ 
0x2f,0x52,0x34,0x4f,0x33,0x50,0x38,0x51,0x0F,0x49,0x37,0x4e,0x0e,0x4a,0x36,0x4d,
0x0c,0x46,0x12,0x47,0x13,0x48,0x2a,0x4c,0x2b,0x39,0x30,0x49,0x1e,0x3a,0x1f,0x3b,
0x20,0x3c,0x21,0x3d,0x22,0x3e,0x23,0x3f,0x24,0x40,0x25,0x41,0x26,0x42,0x27,0x43,
0x2d,0x44,0x2e,0x45,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF
};
#pragma romdata

// EEPROM Map:GHOST(Phantom) Key reduction
#pragma romdata eedata4_scn=0xf000D0
ROM unsigned char eedata4_values[0x10] =
{ 
0b00011111,0b00011111,
0b00011111,0b00011111,
0b00111111,0b00111111,
0b00111111,0b00111111,
0b00111111,0b00011111,
0b00011111,0b00011111,
0b00011111,0b00011111,
0b00011111,0b00011111
};
#pragma romdata

#define FN_KEY         0x68    // HHK like Fn key: 0x68=Caps Lock
#define KMX_MONITOR    0       // 0...Operation  1...Matrix Monitor
#define GHOST_KEY_REDUCTION 1  // 0...Disabled   1...Enabled


/** CONFIGURATION **************************************************/
#if defined(PICDEM_FS_USB)      // Configuration bits for PICDEM FS USB Demo Board (based on PIC18F4550)
        #pragma config PLLDIV   = 5         // (20 MHz crystal on PICDEM FS USB board)
        #pragma config CPUDIV   = OSC1_PLL2   
        #pragma config USBDIV   = 2         // Clock source from 96MHz PLL/2
        #pragma config FOSC     = HSPLL_HS
        #pragma config FCMEN    = OFF
        #pragma config IESO     = OFF
        #pragma config PWRT     = OFF
        #pragma config BOR      = ON
        #pragma config BORV     = 3
        #pragma config VREGEN   = ON      //USB Voltage Regulator
        #pragma config WDT      = OFF
        #pragma config WDTPS    = 32768
        #pragma config MCLRE    = OFF
        #pragma config LPT1OSC  = OFF
        #pragma config PBADEN   = OFF
//      #pragma config CCP2MX   = ON
        #pragma config STVREN   = ON
        #pragma config LVP      = OFF
//      #pragma config ICPRT    = OFF       // Dedicated In-Circuit Debug/Programming
        #pragma config XINST    = OFF       // Extended Instruction Set
        #pragma config CP0      = OFF
        #pragma config CP1      = OFF
//      #pragma config CP2      = OFF
//      #pragma config CP3      = OFF
        #pragma config CPB      = OFF
//      #pragma config CPD      = OFF
        #pragma config WRT0     = OFF
        #pragma config WRT1     = OFF
//      #pragma config WRT2     = OFF
//      #pragma config WRT3     = OFF
        #pragma config WRTB     = OFF       // Boot Block Write Protection
        #pragma config WRTC     = OFF
//      #pragma config WRTD     = OFF
        #pragma config EBTR0    = OFF
        #pragma config EBTR1    = OFF
//      #pragma config EBTR2    = OFF
//      #pragma config EBTR3    = OFF
        #pragma config EBTRB    = OFF
#endif


/** VARIABLES ******************************************************/
#pragma udata
USB_HANDLE lastINTransmission;
USB_HANDLE lastOUTTransmission;
USB_HANDLE lastTransmission;

char ps2_mouse_flg=0;     // 1...PS/2 Mouse data arrived,neet to process
char ps2_mouse_active=0;  // 1...Finish initialize mouse
char ps2_mouse_cnt=0;     // Get 3 Bytes(PS/2 Mouse Packet = 3 Bytes) 
char ps2_clk_cnt=0;       // PS/2 Clock Counter
char ps2_tx_flg=0;        // 1...PS/2 Neet to transmit(Host -> Mouse)
char ps2_mouse=0;         // PS/2 Mouse Byte3 (X data)   /   Tx Byte
char ps2_mouse_2=0;       // PS/2 Mouse Byte2 (Y data)
char ps2_mouse_3=0;       // PS/2 Mouse Byte1 (button,etc...)
signed char mouse_x=0;
signed char mouse_y=0;
signed char mouse_b=0;

/** PRIVATE PROTOTYPES *********************************************/
static void InitializeSystem(void);
void ProcessIO(void);
void UserInit(void);
void YourHighPriorityISRCode();
void YourLowPriorityISRCode();
void Keyboard(void);

void Emulate_Mouse(void);
void USBHIDCBSetReportComplete(void);

/** VECTOR REMAPPING ***********************************************/
#if defined(__18CXX)
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x1000
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x1008
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x1018
	#elif defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)	
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x800
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x808
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x818
	#else	
		#define REMAPPED_RESET_VECTOR_ADDRESS			0x00
		#define REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS	0x08
		#define REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS	0x18
	#endif
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	extern void _startup (void);        // See c018i.c in your C18 compiler dir
	#pragma code REMAPPED_RESET_VECTOR = REMAPPED_RESET_VECTOR_ADDRESS
	void _reset (void)
	{
	    _asm goto _startup _endasm
	}
	#endif
	#pragma code REMAPPED_HIGH_INTERRUPT_VECTOR = REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS
	void Remapped_High_ISR (void)
	{
	     _asm goto YourHighPriorityISRCode _endasm
	}
	#pragma code REMAPPED_LOW_INTERRUPT_VECTOR = REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS
	void Remapped_Low_ISR (void)
	{
	     _asm goto YourLowPriorityISRCode _endasm
	}
	
	#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_MCHPUSB_BOOTLOADER)
	#pragma code HIGH_INTERRUPT_VECTOR = 0x08
	void High_ISR (void)
	{
	     _asm goto REMAPPED_HIGH_INTERRUPT_VECTOR_ADDRESS _endasm
	}
	#pragma code LOW_INTERRUPT_VECTOR = 0x18
	void Low_ISR (void)
	{
	     _asm goto REMAPPED_LOW_INTERRUPT_VECTOR_ADDRESS _endasm
	}
	#endif	//end of "#if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)||defined(PROGRAMMABLE_WITH_USB_LEGACY_CUSTOM_CLASS_BOOTLOADER)"

	#pragma code
	
	
	//These are your actual interrupt handling routines.
	#pragma interrupt YourHighPriorityISRCode
	void YourHighPriorityISRCode()
	{
		//Check which interrupt flag caused the interrupt.
		//Service the interrupt
		//Clear the interrupt flag
		//Etc.
		static char ps2_data=0;
		static char ps2_parity=0;

	    if(PIR2bits.USBIF == 1){
            #if defined(USB_INTERRUPT)
	        USBDeviceTasks();
            #endif
		}
		
	    if(INTCONbits.INT0IF == 1){
		    INTCONbits.INT0IF = 0;
		    
		    if ( ps2_tx_flg==0 ) { // Receive
                ps2_clk_cnt++;
                switch (ps2_clk_cnt) {
                case 1:
            	    if (PORTBbits.RB1==1) ps2_clk_cnt=0;  // Start Bit Error
            	    ps2_data=0;
            	    break;
                case 10:
            	    break;
                case 11:
            	    ps2_clk_cnt=0;
            	    if (PORTBbits.RB1==0) break;      // Stop Bit Error
                    if (ps2_mouse_active) {
                        ps2_mouse_3=ps2_mouse_2;
                        ps2_mouse_2=ps2_mouse;
                        ps2_mouse=ps2_data;
                        ps2_mouse_cnt++;
                        if (ps2_mouse_cnt==3) ps2_mouse_flg=1;
                    }
                    break;
                default:
            	    ps2_data = ps2_data >> 1;
            	    if (PORTBbits.RB1==1) ps2_data|=0b10000000;
            	    break;
                }            
            }            
            else {  // Transmit
                ps2_clk_cnt++;
                switch (ps2_clk_cnt) {
                case 1:
                    ps2_data = ps2_mouse;
                    ps2_parity=0;
                    if ((ps2_data&0b00000001)==0) TRISBbits.TRISB1=0;
                    else {
                        TRISBbits.TRISB1=1;
                        ps2_parity=ps2_parity+1;
                    }
            	    break;
                case 9:
                    ps2_parity++;
                    TRISBbits.TRISB1=(ps2_parity&0b00000001);
            	    break;
                case 10:
                    TRISBbits.TRISB1=1;
            	    break;
                case 11:
                    ps2_clk_cnt=0;
                    ps2_tx_flg=0;
            	    break;
                default:
            	    ps2_data = ps2_data >> 1;
                    if ((ps2_data&0b00000001)==0) TRISBbits.TRISB1=0;
                    else {
                        TRISBbits.TRISB1=1;
                        ps2_parity=ps2_parity+1;
                    }
            	    break;
                }
            }
        }
	}	//This return will be a "retfie fast", since this is in a #pragma interrupt section
	 
	 
	#pragma interruptlow YourLowPriorityISRCode
	void YourLowPriorityISRCode()
	{
		//Check which interrupt flag caused the interrupt.
		//Service the interrupt
		//Clear the interrupt flag
		//Etc.
		static int ps2_timeout=0;
		static int boot_counter=0;
		
		if(PIR1bits.TMR1IF){ 
            PIR1bits.TMR1IF=0;

            WriteTimer1(0xFFFF-128);  // about 100us
            
            if (boot_counter<2000) {boot_counter++;}

            if (boot_counter==800) {
            	CloseRB0INT();
            }
            if (boot_counter==801) {  // about 80ms?
                TRISBbits.TRISB0=0;
                TRISBbits.TRISB1=0;
            }
            if (boot_counter==803) {
                TRISBbits.TRISB0=1;
                TRISBbits.TRISB1=0;
                ps2_tx_flg=1;
                ps2_mouse=0xFF;   // Reset
                ps2_clk_cnt=0;
            	OpenRB0INT(PORTB_CHANGE_INT_ON & PORTB_PULLUPS_ON & FALLING_EDGE_INT);
            }
            
            if (boot_counter==1000) {  // about 100ms?
            	CloseRB0INT();
            }
            if (boot_counter==1001) {
                TRISBbits.TRISB0=0;
                TRISBbits.TRISB1=0;
            }
            if (boot_counter==1003) {
                TRISBbits.TRISB0=1;
                TRISBbits.TRISB1=0;
                ps2_tx_flg=1;
                ps2_mouse=0xF4;  // Enable
                ps2_mouse_active=1;
                ps2_clk_cnt=0;
                ps2_mouse_cnt=0;
            	OpenRB0INT(PORTB_CHANGE_INT_ON & PORTB_PULLUPS_ON & FALLING_EDGE_INT);
            }
            
            ps2_timeout++;
            if (ps2_timeout>1000) {
	            ps2_clk_cnt=0;
	        }
            if (ps2_clk_cnt == 0) ps2_timeout=0;
        }
	}	//This return will be a "retfie", since this is in a #pragma interruptlow section 

#elif defined(__C30__)
    #if defined(PROGRAMMABLE_WITH_USB_HID_BOOTLOADER)
    #endif
#endif




/** DECLARATIONS ***************************************************/
#pragma code
#include <delays.h>

/********************************************************************
 * Function:        void main(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Main program entry point.
 *
 * Note:            None
 *******************************************************************/
#if defined(__18CXX)
void main(void)
#else
int main(void)
#endif
{
    InitializeSystem();

    #if defined(USB_INTERRUPT)
        USBDeviceAttach();
    #endif

    // for PS/2 Mouse CLK
	OpenRB0INT(PORTB_CHANGE_INT_ON & PORTB_PULLUPS_ON & FALLING_EDGE_INT);

    // for PS/2 Mouse initialize
    OpenTimer1(TIMER_INT_ON & T1_16BIT_RW & T1_SOURCE_INT & T1_PS_1_8 & 
            T1_OSC1EN_OFF); 

    RCONbits.IPEN=1;    //***** enable High/Low Interrupt
    IPR1bits.TMR1IP=0;  // Timer1 ... Low Priority interrupt
    INTCONbits.GIEH=1;  // Enable High level interrupt
    INTCONbits.GIEL=1;  // Enable Low  level interrupt

    while(1)
    {
        #if defined(USB_POLLING)
		// Check bus status and service USB interrupts.
        USBDeviceTasks(); // Interrupt or polling method.  If using polling, must call
        #endif
    				  

		// Application-specific tasks.
		// Application related code may be added here, or in the ProcessIO() function.
        ProcessIO();        
    }//end while
}//end main


/********************************************************************
 * Function:        static void InitializeSystem(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        InitializeSystem is a centralize initialization
 *                  routine. All required USB initialization routines
 *                  are called from here.
 *
 *                  User application initialization routine should
 *                  also be called from here.                  
 *
 * Note:            None
 *******************************************************************/
static void InitializeSystem(void)
{
    ADCON1 |= 0x0F;                 // Default all pins to digital

    #if defined(USE_USB_BUS_SENSE_IO)
    tris_usb_bus_sense = INPUT_PIN; // See HardwareProfile.h
    #endif
    
    #if defined(USE_SELF_POWER_SENSE_IO)
    tris_self_power = INPUT_PIN;	// See HardwareProfile.h
    #endif
    
    UserInit();

    USBDeviceInit();	//usb_device.c.  Initializes USB module SFRs and firmware
    					//variables to known states.
}//end InitializeSystem


/******************************************************************************
 * Function:        void UserInit(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This routine should take care of all of the demo code
 *                  initialization that is required.
 *
 * Note:            
 *
 *****************************************************************************/
void UserInit(void)
{
    INTCON2bits.RBPU = 0;
    CMCON=0x7;

    TRISA=0xFF;
    TRISB=0xFF;
    TRISC=0b11111000;
    TRISD=0xFF;
    TRISE=0b01111111;

    PORTA=0x00;
    PORTB=0x00;
    PORTC=0x00;
    PORTD=0x00;
    PORTE=0x00;
    PORTEbits.RDPU = 1;   

    LATA=0x00;
    LATB=0x00;
    LATC=0x00;
    LATD=0x00;
    LATE=0x00;
    
    //initialize the variable holding the handle for the last
    // transmission

    lastINTransmission = 0;
    lastOUTTransmission = 0;

}//end UserInit


/********************************************************************
 * Function:        void ProcessIO(void)
 *	
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is a place holder for other user
 *                  routines. It is a mixture of both USB and
 *                  non-USB tasks.
 *
 * Note:            None
 *******************************************************************/
void ProcessIO(void)
{   
    // User Application USB tasks
    if((USBDeviceState < CONFIGURED_STATE)||(USBSuspendControl==1)) return;

	//Call the function that behaves like a keyboard  
    Keyboard();    
    Emulate_Mouse();
}//end ProcessIO

char eeprom_read(char adress)
{
    EEADR = adress;    
    EECON1bits.EEPGD = 0;
    EECON1bits.CFGS = 0;
    EECON1bits.RD = 1;
    return EEDATA;
}

char kmx2usb(char i) {	
	return eeprom_read(i);
}

char fncnv(char i) {
	char j;	
	for (j=0x80;j<0xC0;j=j+2) {
		if (eeprom_read(j)==i) {
			return eeprom_read(j+1);
		}
	}
	return i;
}

void Keyboard(void)
{
    char i,j,k,p,q;
    char fn_flg0;
    char col1,col2;
    static char fn_flg;
    static char kmx0[17]={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
    static char kmx1[17]={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
    static char kmx2[17]={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
    // Unusable Port:A6,A7,C3,C4,C5,E3
    // Column 00-05: A0-A5
    // Column 06-07: C6-C7
    // Column 08-09: E0-E1
    // Column 10-15: B2-B7
    // NumLk  LED  :C0
    // CapsLk LED  :C1
    // ScrLk  LED  :C2
    // Option      :E2
    // PS/2 CLK    :B0
    // PS/2 DATA   :B1
    col1=0xFE;
    col2=0xFF;
    for (i=0;i<16;i++) {
        TRISA=col1|0b11000000;
        TRISC=(col1|0b00111111)&0b11111000;
        TRISE=col2|0b01111100;
        TRISB=(TRISB|0b11111100)&(col2|0b00000011);

        col2=col2<<1;
        col2|=1;
        if ((col1&0b10000000)==0) col2=0b11111110;
        col1=col1<<1;
        col1|=1;

//        Delay100TCYx(1);    // 0.05us x 4 x 100 = 20us = 0.02ms
        Delay1KTCYx(1);    // 0.05us x 4 x 1000 = 200us = 0.2ms
        k=~PORTD;      
        kmx1[i]=kmx0[i];
        kmx0[i]=0x00;
        for (j=0;j<8;j++) {
            if ((k&(1<<j)) != 0){
                kmx0[i] |= (1<<j);
            }
        }        
    }
	kmx0[16]=0;
	if (PORTEbits.RE2!=1) kmx0[16]|=0b00000001;  // Option Key

#if GHOST_KEY_REDUCTION
    // Ghost (Phantom) Key Reduction
    for (i=0;i<16;i++) {
      kmx2[i]=kmx0[i]&eeprom_read(i+0xD0);
    }
    for (i=0;i<16;i++) {
        k=0;
        for (j=0;j<16;j++) {
            if ( i != j) {
                switch (kmx2[i] & kmx2[j]) {
                case 0:
                case 1:
                case 2:
                case 4:
                case 8:
                case 16:
                case 32:
                case 64:
                case 128:
                    k=1;
                    break;
                default:
                    k=2;
                }
                if ( k==2 ) break;
            }
        }
        if ( k==2 ) kmx0[i]=kmx1[i];    //  It's a GHOST Key !
    }
#endif

    // kmx0(Keyboard Matrix Table) -> hid_report_in(USB TX buffer)
    for (i=0;i<8;i++) {
	    hid_report_in[i]=0;
	}
    q=2;
    fn_flg0=0;
    for (i=0;i<17;i++) {
        for (j=0;j<8;j++) {
            k=i*8+j;
            if ((kmx0[i]&(1<<j)) != 0 ){
	            if (k==FN_KEY) {
		            fn_flg0=1;
		            continue;
		        }
				p=kmx2usb(k);
				if (fn_flg!=0) p=fncnv(p);
				if (p==0xFF) continue;
				if (p>=0xE0&&p<=0xE7) { // Modifier key
					hid_report_in[0] |= 1<<(p-0xE0);
					continue;
				}
				if ( q < 8 ) {
					hid_report_in[q]=p;
					q++;
				}
            }
        }
    }
    fn_flg=fn_flg0;

    if (ps2_mouse_flg==0) {
        mouse_x=0;
        mouse_y=0;
    }

    if (ps2_mouse_flg == 1) {
        ps2_mouse_flg = 0;
        ps2_mouse_cnt = 0;
        mouse_x = ps2_mouse_2;
        mouse_y = -ps2_mouse;
        mouse_b = ps2_mouse_3&0b00000111;
    } 

#if KMX_MONITOR
    for (i=0;i<8;i++) {
	    hid_report_in[i]=0;
	}
    for (i=0;i<17;i++) {
        for (j=0;j<8;j++) {
            k=i*8+j;
            if ((kmx0[i]&(1<<j)) != 0 ){
                p=(k&0b11110000)>>4;
                if (p==0) q=0x27;
                if (p>0 && p<10) q=p+0x1D;
                if (p>=10) q=p-6;
                hid_report_in[2]=q;

                p=(k&0b00001111);
                if (p==0) q=0x27;
                if (p>0 && p<10) q=p+0x1D;
                if (p>=10) q=p-6;
                hid_report_in[3]=q;
	            hid_report_in[4]=0x2c;	//space          
            }
        }
    }
#endif

	//Check if the IN endpoint is not busy, and if it isn't check if we want to send 
	//keystroke data to the host.
    if(!HIDTxHandleBusy(lastINTransmission))
    {
        lastINTransmission = HIDTxPacket(HID_EP, (BYTE*)hid_report_in, 0x08);
    }
    
    //Check if any data was sent from the PC to the keyboard device.  Report descriptor allows
    //host to send 1 byte of data.  Bits 0-4 are LED states, bits 5-7 are unused pad bits.
    //The host can potentially send this OUT report data through the HID OUT endpoint (EP1 OUT),
    //or, alternatively, the host may try to send LED state information by sending a
    //SET_REPORT control transfer on EP0.  See the USBHIDCBSetReportHandler() function.
    if(!HIDRxHandleBusy(lastOUTTransmission))
    {
		//Do something useful with the data now.  Data is in the OutBuffer[0].
		//Num Lock LED state is in Bit0.
        LATC=hid_report_out[0];
		lastOUTTransmission = HIDRxPacket(HID_EP,(BYTE*)&hid_report_out,1);
	} 
}//end keyboard()


/******************************************************************************
 * Function:        void Emulate_Mouse(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    The ownership of the USB buffers will change according
 *                  to the required operation
 *
 * Overview:        This routine will emulate the function of the mouse.  It
 *					does this by sending IN packets of data to the host.
 *					
 *					The generic function HIDTxPacket() is used to send HID
 *					IN packets over USB to the host.
 *
 * Note:            
 *
 *****************************************************************************/
void Emulate_Mouse(void)
{   
    if(HIDTxHandleBusy(lastTransmission) == 0)
    {
        //copy over the data to the HID buffer
        hid_report_in_m[0] = mouse_b;
        hid_report_in_m[1] = mouse_x;
        hid_report_in_m[2] = mouse_y;;
     
        //Send the 3 byte packet over USB to the host.
        lastTransmission = HIDTxPacket(HID_EP_M, (BYTE*)hid_report_in_m, 0x03);
    }
}//end Emulate_Mouse


// ******************************************************************************************************
// ************** USB Callback Functions ****************************************************************
// ******************************************************************************************************
// The USB firmware stack will call the callback functions USBCBxxx() in response to certain USB related
// events.  For example, if the host PC is powering down, it will stop sending out Start of Frame (SOF)
// packets to your device.  In response to this, all USB devices are supposed to decrease their power
// consumption from the USB Vbus to <2.5mA each.  The USB module detects this condition (which according
// to the USB specifications is 3+ms of no bus activity/SOF packets) and then calls the USBCBSuspend()
// function.  You should modify these callback functions to take appropriate actions for each of these
// conditions.  For example, in the USBCBSuspend(), you may wish to add code that will decrease power
// consumption from Vbus to <2.5mA (such as by clock switching, turning off LEDs, putting the
// microcontroller to sleep, etc.).  Then, in the USBCBWakeFromSuspend() function, you may then wish to
// add code that undoes the power saving things done in the USBCBSuspend() function.

// The USBCBSendResume() function is special, in that the USB stack will not automatically call this
// function.  This function is meant to be called from the application firmware instead.  See the
// additional comments near the function.

/******************************************************************************
 * Function:        void USBCBSuspend(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Call back that is invoked when a USB suspend is detected
 *
 * Note:            None
 *****************************************************************************/
void USBCBSuspend(void)
{
    #if defined(__C30__)
    #if 0
        U1EIR = 0xFFFF;
        U1IR = 0xFFFF;
        U1OTGIR = 0xFFFF;
        IFS5bits.USB1IF = 0;
        IEC5bits.USB1IE = 1;
        U1OTGIEbits.ACTVIE = 1;
        U1OTGIRbits.ACTVIF = 1;
        Sleep();
    #endif
    #endif
}


/******************************************************************************
 * Function:        void _USB1Interrupt(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is called when the USB interrupt bit is set
 *					In this example the interrupt is only used when the device
 *					goes to sleep when it receives a USB suspend command
 *
 * Note:            None
 *****************************************************************************/
#if 0
void __attribute__ ((interrupt)) _USB1Interrupt(void)
{
    #if !defined(self_powered)
        if(U1OTGIRbits.ACTVIF)
        {       
            IEC5bits.USB1IE = 0;
            U1OTGIEbits.ACTVIE = 0;
            IFS5bits.USB1IF = 0;
        
            //USBClearInterruptFlag(USBActivityIFReg,USBActivityIFBitNum);
            USBClearInterruptFlag(USBIdleIFReg,USBIdleIFBitNum);
            //USBSuspendControl = 0;
        }
    #endif
}
#endif

/******************************************************************************
 * Function:        void USBCBWakeFromSuspend(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The host may put USB peripheral devices in low power
 *					suspend mode (by "sending" 3+ms of idle).  Once in suspend
 *					mode, the host may wake the device back up by sending non-
 *					idle state signalling.
 *					
 *					This call back is invoked when a wakeup from USB suspend 
 *					is detected.
 *
 * Note:            None
 *****************************************************************************/
void USBCBWakeFromSuspend(void)
{
}

/********************************************************************
 * Function:        void USBCB_SOF_Handler(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The USB host sends out a SOF packet to full-speed
 *                  devices every 1 ms. This interrupt may be useful
 *                  for isochronous pipes. End designers should
 *                  implement callback routine as necessary.
 *
 * Note:            None
 *******************************************************************/
void USBCB_SOF_Handler(void)
{
    // No need to clear UIRbits.SOFIF to 0 here.
    // Callback caller is already doing that.
}

/*******************************************************************
 * Function:        void USBCBErrorHandler(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The purpose of this callback is mainly for
 *                  debugging during development. Check UEIR to see
 *                  which error causes the interrupt.
 *
 * Note:            None
 *******************************************************************/
void USBCBErrorHandler(void)
{
}

/*******************************************************************
 * Function:        void USBCBCheckOtherReq(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        When SETUP packets arrive from the host, some
 * 					firmware must process the request and respond
 *					appropriately to fulfill the request.  Some of
 *					the SETUP packets will be for standard
 *					USB "chapter 9" (as in, fulfilling chapter 9 of
 *					the official USB specifications) requests, while
 *					others may be specific to the USB device class
 *					that is being implemented.  For example, a HID
 *					class device needs to be able to respond to
 *					"GET REPORT" type of requests.  This
 *					is not a standard USB chapter 9 request, and 
 *					therefore not handled by usb_device.c.  Instead
 *					this request should be handled by class specific 
 *					firmware, such as that contained in usb_function_hid.c.
 *
 * Note:            None
 *******************************************************************/
void USBCBCheckOtherReq(void)
{
//    USBCheckHIDRequest();
    switch ( SetupPkt.wIndex ) {
        case HID_INTF_ID:   USBCheckHIDRequest()  ;break;
        case HID_INTF_ID_M: USBCheckHIDRequest_m();break;
        default:                                   break;
    }       
}//end


/*******************************************************************
 * Function:        void USBCBStdSetDscHandler(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The USBCBStdSetDscHandler() callback function is
 *					called when a SETUP, bRequest: SET_DESCRIPTOR request
 *					arrives.  Typically SET_DESCRIPTOR requests are
 *					not used in most applications, and it is
 *					optional to support this type of request.
 *
 * Note:            None
 *******************************************************************/
void USBCBStdSetDscHandler(void)
{
    // Must claim session ownership if supporting this request
}//end


/*******************************************************************
 * Function:        void USBCBInitEP(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is called when the device becomes
 *                  initialized, which occurs after the host sends a
 * 					SET_CONFIGURATION (wValue not = 0) request.  This 
 *					callback function should initialize the endpoints 
 *					for the device's usage according to the current 
 *					configuration.
 *
 * Note:            None
 *******************************************************************/
void USBCBInitEP(void)
{
    //enable the HID endpoint
    USBEnableEndpoint(HID_EP,USB_IN_ENABLED|USB_OUT_ENABLED|USB_HANDSHAKE_ENABLED|USB_DISALLOW_SETUP);
    USBEnableEndpoint(HID_EP_M,USB_IN_ENABLED|USB_OUT_ENABLED|USB_HANDSHAKE_ENABLED|USB_DISALLOW_SETUP);

	lastOUTTransmission = HIDRxPacket(HID_EP,(BYTE*)&hid_report_out,1);
}

/********************************************************************
 * Function:        void USBCBSendResume(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        The USB specifications allow some types of USB
 * 					peripheral devices to wake up a host PC (such
 *					as if it is in a low power suspend to RAM state).
 *					This can be a very useful feature in some
 *					USB applications, such as an Infrared remote
 *					control	receiver.  If a user presses the "power"
 *					button on a remote control, it is nice that the
 *					IR receiver can detect this signalling, and then
 *					send a USB "command" to the PC to wake up.
 *					
 *					The USBCBSendResume() "callback" function is used
 *					to send this special USB signalling which wakes 
 *					up the PC.  This function may be called by
 *					application firmware to wake up the PC.  This
 *					function should only be called when:
 *					
 *					1.  The USB driver used on the host PC supports
 *						the remote wakeup capability.
 *					2.  The USB configuration descriptor indicates
 *						the device is remote wakeup capable in the
 *						bmAttributes field.
 *					3.  The USB host PC is currently sleeping,
 *						and has previously sent your device a SET 
 *						FEATURE setup packet which "armed" the
 *						remote wakeup capability.   
 *
 *					This callback should send a RESUME signal that
 *                  has the period of 1-15ms.
 *
 * Note:            Interrupt vs. Polling
 *                  -Primary clock
 *                  -Secondary clock ***** MAKE NOTES ABOUT THIS *******
 *                   > Can switch to primary first by calling USBCBWakeFromSuspend()
 
 *                  The modifiable section in this routine should be changed
 *                  to meet the application needs. Current implementation
 *                  temporary blocks other functions from executing for a
 *                  period of 1-13 ms depending on the core frequency.
 *
 *                  According to USB 2.0 specification section 7.1.7.7,
 *                  "The remote wakeup device must hold the resume signaling
 *                  for at lest 1 ms but for no more than 15 ms."
 *                  The idea here is to use a delay counter loop, using a
 *                  common value that would work over a wide range of core
 *                  frequencies.
 *                  That value selected is 1800. See table below:
 *                  ==========================================================
 *                  Core Freq(MHz)      MIP         RESUME Signal Period (ms)
 *                  ==========================================================
 *                      48              12          1.05
 *                       4              1           12.6
 *                  ==========================================================
 *                  * These timing could be incorrect when using code
 *                    optimization or extended instruction mode,
 *                    or when having other interrupts enabled.
 *                    Make sure to verify using the MPLAB SIM's Stopwatch
 *                    and verify the actual signal on an oscilloscope.
 *******************************************************************/
void USBCBSendResume(void)
{
    static WORD delay_count;
    
    USBResumeControl = 1;                // Start RESUME signaling
    
    delay_count = 1800U;                // Set RESUME line for 1-13 ms
    do
    {
        delay_count--;
    }while(delay_count);
    USBResumeControl = 0;
}


/*******************************************************************
 * Function:        BOOL USER_USB_CALLBACK_EVENT_HANDLER(
 *                        USB_EVENT event, void *pdata, WORD size)
 *
 * PreCondition:    None
 *
 * Input:           USB_EVENT event - the type of event
 *                  void *pdata - pointer to the event data
 *                  WORD size - size of the event data
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        This function is called from the USB stack to
 *                  notify a user application that a USB event
 *                  occured.  This callback is in interrupt context
 *                  when the USB_INTERRUPT option is selected.
 *
 * Note:            None
 *******************************************************************/
BOOL USER_USB_CALLBACK_EVENT_HANDLER(USB_EVENT event, void *pdata, WORD size)
{
    switch(event)
    {
        case EVENT_CONFIGURED: 
            USBCBInitEP();
            break;
        case EVENT_SET_DESCRIPTOR:
            USBCBStdSetDscHandler();
            break;
        case EVENT_EP0_REQUEST:
            USBCBCheckOtherReq();
            break;
        case EVENT_SOF:
            USBCB_SOF_Handler();
            break;
        case EVENT_SUSPEND:
            USBCBSuspend();
            break;
        case EVENT_RESUME:
            USBCBWakeFromSuspend();
            break;
        case EVENT_BUS_ERROR:
            USBCBErrorHandler();
            break;
        case EVENT_TRANSFER:
            Nop();
            break;
        default:
            break;
    }      
    return TRUE; 
}


// *****************************************************************************
// ************** USB Class Specific Callback Function(s) **********************
// *****************************************************************************

/********************************************************************
 * Function:        void USBHIDCBSetReportHandler(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        USBHIDCBSetReportHandler() is used to respond to
 *					the HID device class specific SET_REPORT control
 *					transfer request (starts with SETUP packet on EP0 OUT).  
 * Note:            
 *******************************************************************/
void USBHIDCBSetReportHandler(void)
{
	//Prepare to receive the keyboard LED state data through a SET_REPORT
	//control transfer on endpoint 0.  The host should only send 1 byte,
	//since this is all that the report descriptor allows it to send.
	USBEP0Receive((BYTE*)&CtrlTrfData, USB_EP0_BUFF_SIZE, USBHIDCBSetReportComplete);
}

//Secondary callback function that gets called when the above
//control transfer completes for the USBHIDCBSetReportHandler()
void USBHIDCBSetReportComplete(void)
{

}	
/** EOF Keyboard.c **********************************************/
#endif
